
#include <windows.h>
#include <ddraw.h>
#include <mmsystem.h>
#include "errcode.h"

//Function prototypes in dxdraw.h conflict with extern "C" specifications
#define NO_DXDRAW_FUNCTION_PROTOTYPES
#include "dxdraw.h"
#undef NO_DXDRAW_FUNCTION_PROTOTYPES

HINSTANCE hDirectDraw = NULL;
IDirectDraw *pDirectDraw = NULL;
IDirectDrawSurface *pPrimaryDDS = NULL;
IDirectDrawSurface *pOverlayDDS = NULL;
IDirectDrawSurface *pOverlayBackBufferDDS = NULL;
DDSURFACEDESC overlayDDSD;
RECT overlaySourceRect, overlayDestRect;
LONG glOverlayWidth, glOverlayHeight;
HRESULT ghDxDrawErrorCode;
BOOL gbDoubleBufferedOverlay = FALSE;
HWND ghWnd;

IDirectDrawSurface *pOffScreenDDS = NULL;
DDSURFACEDESC offScreenDDSD;
IDirectDrawClipper *pClipper = NULL;

extern "C" LONG InitializeDirectDraw(VOID);
extern "C" VOID DestroyDirectDraw(VOID);
extern "C" VOID UnloadDirectDraw(VOID);
extern "C" LONG SetNormalCooperativeLevel(HWND);
extern "C" LONG CheckOverlayCaps(DWORD, DWORD, DWORD *, DWORD *);
extern "C" LONG CreateOverlay(DWORD, DWORD, BOOL, DWORD, BOOL, FOURCC, DWORD, HWND);
extern "C" VOID FreeOverlay(VOID);
extern "C" LONG FlipOverlay(VOID);
extern "C" LONG LockOverlay(LPVOID *, DWORD *);
extern "C" LONG UnlockOverlay(VOID);
extern "C" LONG MoveOverlay(LONG, LONG);
extern "C" LONG ShowOverlay(LONG, LONG, DWORD, COLORREF);
extern "C" LONG HideOverlay(VOID);
extern "C" LONG GetOverlayWidth(VOID);
extern "C" LONG GetOverlayHeight(VOID);
extern "C" HRESULT GetDxDrawErrorCode(VOID);
extern "C" VOID SetDxDrawErrorCode(HRESULT);

DWORD CalculateColorKey(IDirectDrawSurface *, COLORREF);

extern "C" LONG CreateOffScreenSurface(DWORD, DWORD, BOOL, DWORD, BOOL, FOURCC, HWND);
extern "C" VOID FreeOffScreenSurface(VOID);
extern "C" LONG FlipOffScreenSurface(VOID);
extern "C" LONG LockOffScreenSurface(LPVOID *, DWORD *);
extern "C" LONG UnlockOffScreenSurface(VOID);

LONG InitializeDirectDraw(VOID) {
	//Must be static so that this pointer to a function remains valid
	//each time InitDirectDraw is called
	static HRESULT (__stdcall *pDirectDrawCreate)
		(GUID FAR *, LPDIRECTDRAW FAR *, IUnknown FAR *) = NULL;
	HRESULT hResult;

	//Try to load ddraw.dll if it has not already been loaded
	if (!hDirectDraw) {
		hDirectDraw = LoadLibrary("ddraw.dll");
		
		//See if the library was loaded successfully
		if (!hDirectDraw) {
			return EC_FAILED_TO_LOAD_DIRECTDRAW_LIBRARY;
		}

		//Get the address of DirectDrawCreate
		pDirectDrawCreate = (HRESULT (__stdcall *)
			(GUID FAR *, LPDIRECTDRAW FAR *, IUnknown FAR *))
			GetProcAddress(hDirectDraw, "DirectDrawCreate");

		//See if the procedure address was found
		if (!pDirectDrawCreate) {
			//If the library was loaded (hDirectDraw != NULL), all the
			//procedure addresses should be valid
			//Since this procedure address was not found, unload the library
			UnloadDirectDraw();
			return EC_FAILED_TO_GET_DIRECTDRAWCREATE_PROCEDURE_ADDRESS;
		}
	}

	//Return OK if the surface has already been created
	if (pDirectDraw) {
		return EC_OK;
	}

	//Create a DirectDraw object
	hResult = (*pDirectDrawCreate)(NULL, &pDirectDraw, NULL);

	//See if the DirectDraw object was created successfully
	if (hResult != DD_OK) {
		return EC_DIRECTDRAWCREATE_FAILED;
	}

	//Return OK
	return EC_OK;
}

VOID DestroyDirectDraw(VOID) {
	if (pDirectDraw) {
		pDirectDraw->Release();
		pDirectDraw = NULL;
	}
	return;
}

VOID UnloadDirectDraw(VOID) {
	//Unload ddraw.dll if it has been loaded
	if (hDirectDraw) {
		//TODO: maybe check that this function worked and log errors
		FreeLibrary(hDirectDraw);
		hDirectDraw = NULL;
	}
	return;
}

LONG SetNormalCooperativeLevel(HWND hWnd) {
	HRESULT hResult;

	hResult = pDirectDraw->SetCooperativeLevel(hWnd, DDSCL_NORMAL);
	if (hResult != DD_OK) {
		return EC_SETCOOPERATIVELEVEL_FAILED;
	}
	return EC_OK;
}

LONG CheckOverlayCaps(DWORD dwWidth, DWORD dwHeight, DWORD *pdwMinStretch,
	DWORD *pdwClippingRecommendation) {

	HRESULT hResult;
	DDCAPS sDDCAPS;
	int iFullScreenWidth, iFullScreenHeight;

	//Set up the DDCAPS structure
	ZeroMemory(&sDDCAPS, sizeof(sDDCAPS));
	sDDCAPS.dwSize = sizeof(sDDCAPS);

	//Get the caps
	hResult = pDirectDraw->GetCaps(&sDDCAPS, NULL);
	if (hResult != DD_OK) {
		//Set DxDraw error code
		SetDxDrawErrorCode(hResult);
		return EC_COULD_NOT_GET_DIRECTDRAW_CAPS;
	}

	//Examine the caps
	//See if overlays are supported
	if (!(sDDCAPS.dwCaps & DDCAPS_OVERLAY)) {
		return EC_NO_OVERLAY_CAPS_DETECTED;
	}

	//Get the full screen dimensions
	iFullScreenWidth = GetSystemMetrics(SM_CXFULLSCREEN);
	iFullScreenHeight = GetSystemMetrics(SM_CYFULLSCREEN);

	//See if the overlay hardware is capable of stretching
	if (sDDCAPS.dwCaps & DDCAPS_OVERLAYSTRETCH) {
		DWORD dwMinOverlayStretch;
		DWORD dwMinOverlayWidth;
		
		//Get the minimum overlay stretch factor
		dwMinOverlayStretch = sDDCAPS.dwMinOverlayStretch;

		//Calculate the minimum width of the window needed to display the overlay
		dwMinOverlayWidth = (dwWidth * dwMinOverlayStretch) / 1000;

		//Return the minimum overlay stretch factor
		*pdwMinStretch = dwMinOverlayStretch;

		//See if the minimum width of the window needed to display the overlay will fit on the screen
		if (dwMinOverlayWidth > (DWORD)iFullScreenWidth) {
			return EC_MINIMUM_OVERLAY_STRETCH_FACTOR_TOO_LARGE;
		}
	}

	//See if overlay clipping is supported
	if (sDDCAPS.dwCaps & DDCAPS_OVERLAYCANTCLIP) {
		*pdwClippingRecommendation = OVERLAY_ENABLE_COLOR_KEY_CLIPPING;
	}
	else {
		*pdwClippingRecommendation = OVERLAY_ENABLE_WINDOW_CLIPPING;
	}

	return EC_OK;
}

LONG CreateOverlay(DWORD dwWidth, DWORD dwHeight, BOOL bDoubleBuffering, DWORD dwBackBufferCount,
	BOOL bUseFourCCSurface, FOURCC fourCC, DWORD dwOverlayFlags, HWND hWnd) {

	HRESULT hResult;
	DDSURFACEDESC sDDSD;
	DDSCAPS sDDSCAPS;

	//Create a primary surface
	ZeroMemory(&sDDSD, sizeof(sDDSD));
	sDDSD.dwSize = sizeof(sDDSD);
	sDDSD.dwFlags = DDSD_CAPS;
	sDDSD.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

	hResult = pDirectDraw->CreateSurface(&sDDSD, &pPrimaryDDS, NULL);
	if (hResult != DD_OK) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(hResult);

		return EC_COULD_NOT_CREATE_PRIMARY_SURFACE;
	}
	
	//Create the overlay
	ZeroMemory(&overlayDDSD, sizeof(overlayDDSD));
	overlayDDSD.dwSize = sizeof(overlayDDSD);
	overlayDDSD.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT | DDSD_PIXELFORMAT;
	overlayDDSD.dwWidth = dwWidth;
	overlayDDSD.dwHeight = dwHeight;
	
	ZeroMemory(&overlayDDSD.ddpfPixelFormat, sizeof(overlayDDSD.ddpfPixelFormat));
	overlayDDSD.ddpfPixelFormat.dwSize = sizeof(overlayDDSD.ddpfPixelFormat);

	if (bUseFourCCSurface == TRUE) {
		//FourCC YUV surface
		overlayDDSD.ddpfPixelFormat.dwFlags = DDPF_FOURCC;
		overlayDDSD.ddpfPixelFormat.dwFourCC = fourCC;
		overlayDDSD.ddpfPixelFormat.dwYUVBitCount = 16;
	}
	else {
		//RGB surface
		overlayDDSD.ddpfPixelFormat.dwFlags = DDPF_RGB;
		overlayDDSD.ddpfPixelFormat.dwRGBBitCount = 16;
		overlayDDSD.ddpfPixelFormat.dwRBitMask = 0x0000F800;
		overlayDDSD.ddpfPixelFormat.dwGBitMask = 0x000007E0;
		overlayDDSD.ddpfPixelFormat.dwBBitMask = 0x0000001F;
	}

	overlayDDSD.ddsCaps.dwCaps = DDSCAPS_OVERLAY | DDSCAPS_VIDEOMEMORY;


	//Default to a single buffered overlay
	gbDoubleBufferedOverlay = FALSE;

	//See if double buffering is enabled
	if (bDoubleBuffering == TRUE) {
		overlayDDSD.dwFlags |= DDSD_BACKBUFFERCOUNT;

		//See how many back buffers are being requested
		if (dwBackBufferCount == 2) {
			overlayDDSD.dwBackBufferCount = 2;
		}
		else {
			overlayDDSD.dwBackBufferCount = 1;
		}

		overlayDDSD.ddsCaps.dwCaps |= DDSCAPS_FLIP | DDSCAPS_COMPLEX;

		//Using a double buffered overlay
		gbDoubleBufferedOverlay = TRUE;
	}

	//Try to create the overlay surface
	hResult = pDirectDraw->CreateSurface(&overlayDDSD, &pOverlayDDS, NULL);

	//See if the overlay surface was created successfully
	if (hResult != DD_OK) {
		//Set the surface pointer to NULL
		pOverlayDDS = NULL;

		//Set DxDraw error code
		SetDxDrawErrorCode(hResult);
		return EC_COULD_NOT_CREATE_OVERLAY_SURFACE;
	}

	//Get the attached back buffer if double buffering is enabled
	if (bDoubleBuffering == TRUE) {
		sDDSCAPS.dwCaps = DDSCAPS_BACKBUFFER;
		hResult = pOverlayDDS->GetAttachedSurface(&sDDSCAPS, &pOverlayBackBufferDDS);

		//Make sure the get back buffer surface call succeeded
		if (hResult != DD_OK) {
			//Set the surface pointer to NULL
			pOverlayBackBufferDDS = NULL;

			//Release the overlay surface
			pOverlayDDS->Release();
			pOverlayDDS = NULL;

			//Set DxDraw error code
			SetDxDrawErrorCode(hResult);
			return EC_COULD_NOT_GET_BACK_BUFFER;
		}
	}
	
	//See if window clipping is enabled
	if ((dwOverlayFlags & OVERLAY_ENABLE_WINDOW_CLIPPING) != 0) {
		//Create a clipper
		hResult = pDirectDraw->CreateClipper(0, &pClipper, NULL);
		//See if the clipper was create successfully
		if (hResult != DD_OK) {
			return EC_COULD_NOT_CREATE_CLIPPER;
		}

		//Set the clipping window
		pClipper->SetHWnd(0, hWnd);
		//See if the clipping window was set successfully
		if (hResult != DD_OK) {
			//Release the clipper
			pClipper->Release();
			pClipper = NULL;

			return EC_COULD_NOT_SET_CLIPPING_WINDOW;
		}

		//Attach the clipper
		hResult = pPrimaryDDS->SetClipper(pClipper);
		//See if the clipper was attached successfully
		if (hResult != DD_OK) {
			//Release the clipper
			pClipper->Release();
			pClipper = NULL;

			return EC_COULD_NOT_ATTACH_CLIPPER;
		}
	}
	
	//Set up the source rectangle
	overlaySourceRect.left = 0;
	overlaySourceRect.top = 0;
	overlaySourceRect.right = dwWidth;
	overlaySourceRect.bottom = dwHeight;

	//Clear the overlay before showing it
	//Lock the overlay
	ZeroMemory(&sDDSD, sizeof(sDDSD));
	sDDSD.dwSize = sizeof(sDDSD);
	hResult = pOverlayDDS->Lock(NULL, &sDDSD, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	
	if (hResult == DD_OK) {
		DWORD i;
		BYTE *pCurrentScanline;

		//Clear the overlay
		pCurrentScanline = (BYTE *)sDDSD.lpSurface;
		for (i = 0; i < dwHeight; i++) {
			//Zero a scanline (2 bytes per pixel)
			ZeroMemory(pCurrentScanline, dwWidth << 2);
			pCurrentScanline += sDDSD.lPitch;
		}
		
		//Unlock the overlay
		hResult = pOverlayDDS->Unlock(NULL);
	}

	return EC_OK;
}

VOID FreeOverlay(VOID) {
	//Turn the overlay off
	HideOverlay();

	//Detach and release the clipper
	if (pClipper != NULL) {
		//Detach the clipper
		pPrimaryDDS->SetClipper(NULL);

		//Release the clipper
		pClipper->Release();
		pClipper = NULL;
	}

	//Release the back buffer overlay surface
	if (pOverlayBackBufferDDS != NULL) {
		pOverlayBackBufferDDS->Release();
		pOverlayBackBufferDDS = NULL;
	}

	//Release the overlay surface
	if (pOverlayDDS != NULL) {
		pOverlayDDS->Release();
		pOverlayDDS = NULL;
	}

	//Release the primary surface
	if (pPrimaryDDS != NULL) {
		pPrimaryDDS->Release();
		pPrimaryDDS = NULL;
	}

	return;
}

LONG FlipOverlay(VOID) {
	HRESULT hResult;

	//Flip the overlay surface
	if (pOverlayDDS != NULL) {
		hResult = pOverlayDDS->Flip(NULL, DDFLIP_WAIT);
	}
	return EC_OK;
}

LONG LockOverlay(LPVOID *ppSurface, DWORD *pdwSurfacePitch) {
	HRESULT hResult;
	DDSURFACEDESC sDDSD;

	//Make sure the surface exists
	if (pOverlayDDS == NULL) {
		return EC_COULD_NOT_LOCK_SURFACE;
	}

	//See if double buffering is enabled
	if (gbDoubleBufferedOverlay == TRUE) {
		ZeroMemory(&sDDSD, sizeof(sDDSD));
		sDDSD.dwSize = sizeof(sDDSD);
		hResult = pOverlayBackBufferDDS->Lock(NULL, &sDDSD, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	}
	else {
		ZeroMemory(&sDDSD, sizeof(sDDSD));
		sDDSD.dwSize = sizeof(sDDSD);
		hResult = pOverlayDDS->Lock(NULL, &sDDSD, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	}

	if (hResult != DD_OK) {
		return EC_COULD_NOT_LOCK_SURFACE;
	}

	//Return data about the surface
	*ppSurface = sDDSD.lpSurface;
	*pdwSurfacePitch = sDDSD.lPitch;

	return EC_OK;
}

LONG UnlockOverlay(VOID) {
	HRESULT hResult;

	//See if double buffering is enabled
	if (gbDoubleBufferedOverlay == TRUE) {
		if (pOverlayBackBufferDDS != NULL) {
			hResult = pOverlayBackBufferDDS->Unlock(NULL);
		}
	}
	else {
		if (pOverlayDDS != NULL) {
			hResult = pOverlayDDS->Unlock(NULL);
		}
	}
	return EC_OK;
}

LONG MoveOverlay(LONG lXPos, LONG lYPos) {
	HRESULT hResult;

	//Make sure the pointer to the surface is valid
	if (pOverlayDDS == NULL) {
		return EC_COULD_NOT_MOVE_OVERLAY;
	}

	//Move the overlay
	hResult = pOverlayDDS->SetOverlayPosition(lXPos, lYPos);

	if (hResult != DD_OK) {
		return EC_COULD_NOT_MOVE_OVERLAY;
	}
	return EC_OK;
}

LONG ShowOverlay(LONG lWidth, LONG lHeight, DWORD dwOverlayFlags, COLORREF colorKey) {
	HRESULT hResult;
	DDCOLORKEY sDDCK;
	DDOVERLAYFX sDDOFX;
	DWORD dwRet;
	DWORD dwFlags;
	LONG lRet;
	DWORD dwRestoreAttempts;
	
	if (pOverlayDDS == NULL) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(DDERR_INVALIDOBJECT);

		return EC_COULD_NOT_SHOW_OVERLAY;
	}

	//Set the overlay's destination rectangle
	overlayDestRect.left = 0;
	overlayDestRect.top = 0;
	overlayDestRect.right = lWidth;
	overlayDestRect.bottom = lHeight;

	//Setup the overlay description
	ZeroMemory(&sDDOFX, sizeof(sDDOFX));
	sDDOFX.dwSize = sizeof(sDDOFX);

	//Setup the flags
	dwFlags = DDOVER_SHOW;
	
	if ((dwOverlayFlags & OVERLAY_ENABLE_COLOR_KEY_CLIPPING) != 0) {
		//Setup color keying
		dwRet = CalculateColorKey(pPrimaryDDS, colorKey);
		if (dwRet == CLR_INVALID) {
			//Return an error
			return EC_COULD_NOT_CALCULATE_COLOR_KEY;
		}
		sDDCK.dwColorSpaceLowValue = dwRet;
		sDDCK.dwColorSpaceHighValue = dwRet;
		sDDOFX.dckDestColorkey = sDDCK;

		//Enable destination color keying
		dwFlags |= DDOVER_KEYDESTOVERRIDE;
	}
	
	if ((dwOverlayFlags & OVERLAY_ENABLE_ALPHA_BLENDING) != 0) {
		//Setup alpha blending
		sDDOFX.dwAlphaDestConstBitDepth = 8;
		sDDOFX.dwAlphaDestConst = 128;

		//Enable constant destination alpha blending
		dwFlags |= DDOVER_ALPHADESTCONSTOVERRIDE;
	}
	
	//Show the overlay
	dwRestoreAttempts = MAX_SURFACE_RESTORE_ATTEMPTS;
	do {
		//Show the overlay
		hResult = pOverlayDDS->UpdateOverlay(&overlaySourceRect,
			pPrimaryDDS, &overlayDestRect, dwFlags, &sDDOFX);

		//Count the attempts to restore the surface
		if (dwRestoreAttempts == 0) {
			break;
		}
		dwRestoreAttempts--;

		//Restore the surface if it was lost
		if (hResult == DDERR_SURFACELOST) {
			pOverlayDDS->Restore();
		}
	} while (hResult == DDERR_SURFACELOST);


	if (hResult != DD_OK) {
		//Hide the overlay
		lRet = HideOverlay();

		//Set the DxDraw error code
		SetDxDrawErrorCode(hResult);
		return EC_COULD_NOT_SHOW_OVERLAY;
	}

	//Save the size of the overlay
	glOverlayWidth = lWidth;
	glOverlayHeight = lHeight;

	return EC_OK;
}

LONG HideOverlay(VOID) {
	if (pOverlayDDS != NULL) {
		pOverlayDDS->UpdateOverlay(NULL, pPrimaryDDS, NULL, DDOVER_HIDE, NULL);
	}
	return EC_OK;
}

LONG GetOverlayWidth(VOID) {
	if (pOverlayDDS != NULL) {
		return glOverlayWidth;
	}
	return 0;
}

LONG GetOverlayHeight(VOID) {
	if (pOverlayDDS != NULL) {
		return glOverlayHeight;
	}
	return 0;
}

HRESULT GetDxDrawErrorCode(VOID) {
	return ghDxDrawErrorCode;
}

VOID SetDxDrawErrorCode(HRESULT hResult) {
	ghDxDrawErrorCode = hResult;
	return;
}

DWORD CalculateColorKey(IDirectDrawSurface *pDDS, COLORREF rgb) {
    COLORREF oldPixel;
    HDC hDC;
    DWORD dw = CLR_INVALID;
    DDSURFACEDESC sDDSD;
    HRESULT hResult;
	DWORD dwRestoreAttempts;

	//Set a pixel using GDI
    if (pDDS->GetDC(&hDC) == DD_OK) {
		//Save the old pixel
        oldPixel = GetPixel(hDC, 0, 0);

		//Set the pixel
        SetPixel(hDC, 0, 0, rgb);

        pDDS->ReleaseDC(hDC);
    }

	//Lock the surface so the color of the pixel can be read back
	dwRestoreAttempts = MAX_SURFACE_RESTORE_ATTEMPTS;
	do {
		//Lock the surface
		sDDSD.dwSize = sizeof(sDDSD);
		while ((hResult = pDDS->Lock(NULL, &sDDSD, 0, NULL)) == DDERR_WASSTILLDRAWING);

		//Count the attempts to restore the surface
		if (dwRestoreAttempts == 0) {
			break;
		}
		dwRestoreAttempts--;

		//Restore the surface if it was lost
		if (hResult == DDERR_SURFACELOST) {
			pDDS->Restore();
		}
	} while (hResult == DDERR_SURFACELOST);


    if (hResult == DD_OK) {
        dw = *(DWORD *)sDDSD.lpSurface;
        dw &= (1 << sDDSD.ddpfPixelFormat.dwRGBBitCount) - 1;
        pDDS->Unlock(NULL);
    }

	//Put the old pixel back
    if (pDDS->GetDC(&hDC) == DD_OK) {
        SetPixel(hDC, 0, 0, oldPixel);
        pDDS->ReleaseDC(hDC);
    }

    return dw;
}

LONG CreateOffScreenSurface(DWORD dwWidth, DWORD dwHeight, BOOL bDoubleBuffering, DWORD dwBackBufferCount,
	BOOL bUseFourCCSurface, FOURCC fourCC, HWND hWnd) {

	HRESULT hResult;
	DDSURFACEDESC sDDSD;

	//Set up the global hWnd
	ghWnd = hWnd;

	//Create a primary surface
	ZeroMemory(&sDDSD, sizeof(sDDSD));
	sDDSD.dwSize = sizeof(sDDSD);
	sDDSD.dwFlags = DDSD_CAPS;
	sDDSD.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

	hResult = pDirectDraw->CreateSurface(&sDDSD, &pPrimaryDDS, NULL);
	if (hResult != DD_OK) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(hResult);

		return EC_COULD_NOT_CREATE_PRIMARY_SURFACE;
	}
	
	//Create the off screen surface
	ZeroMemory(&overlayDDSD, sizeof(overlayDDSD));
	overlayDDSD.dwSize = sizeof(overlayDDSD);
	overlayDDSD.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT | DDSD_PIXELFORMAT;
	overlayDDSD.dwWidth = dwWidth;
	overlayDDSD.dwHeight = dwHeight;
	
	ZeroMemory(&overlayDDSD.ddpfPixelFormat, sizeof(overlayDDSD.ddpfPixelFormat));
	overlayDDSD.ddpfPixelFormat.dwSize = sizeof(overlayDDSD.ddpfPixelFormat);

	if (bUseFourCCSurface == TRUE) {
		//FourCC YUV surface
		overlayDDSD.ddpfPixelFormat.dwFlags = DDPF_FOURCC;
		overlayDDSD.ddpfPixelFormat.dwFourCC = fourCC;
		overlayDDSD.ddpfPixelFormat.dwYUVBitCount = 16;
	}
	else {
		//RGB surface
		//RGB surfaces are not supported by this version of WinGlide

		SetDxDrawErrorCode(DDERR_INVALIDPIXELFORMAT);
		return EC_COULD_NOT_CREATE_OFF_SCREEN_SURFACE;

/*		overlayDDSD.ddpfPixelFormat.dwFlags = DDPF_RGB;
		overlayDDSD.ddpfPixelFormat.dwRGBBitCount = 16;
		overlayDDSD.ddpfPixelFormat.dwRBitMask = 0x0000F800;
		overlayDDSD.ddpfPixelFormat.dwGBitMask = 0x000007E0;
		overlayDDSD.ddpfPixelFormat.dwBBitMask = 0x0000001F;*/
	}

	overlayDDSD.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY;


	//Default to a single buffered off screen surface
	gbDoubleBufferedOverlay = FALSE;


	//Try to create the off screen surface
	hResult = pDirectDraw->CreateSurface(&overlayDDSD, &pOverlayDDS, NULL);

	//See if the off screen surface was created successfully
	if (hResult != DD_OK) {
		//Set the surface pointer to NULL
		pOverlayDDS = NULL;

		//Set DxDraw error code
		SetDxDrawErrorCode(hResult);
		return EC_COULD_NOT_CREATE_OFF_SCREEN_SURFACE;
	}

	//Set up the source rectangle
	overlaySourceRect.left = 0;
	overlaySourceRect.top = 0;
	overlaySourceRect.right = dwWidth;
	overlaySourceRect.bottom = dwHeight;

	//Clear the off screen surface before showing it
	//Lock the off screen surface
	ZeroMemory(&sDDSD, sizeof(sDDSD));
	sDDSD.dwSize = sizeof(sDDSD);
	hResult = pOverlayDDS->Lock(NULL, &sDDSD, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	
	if (hResult == DD_OK) {
		DWORD i;
		BYTE *pCurrentScanline;

		//Clear the off screen surface
		pCurrentScanline = (BYTE *)sDDSD.lpSurface;
		for (i = 0; i < dwHeight; i++) {
			//Zero a scanline (2 bytes per pixel)
			ZeroMemory(pCurrentScanline, dwWidth << 2);
			pCurrentScanline += sDDSD.lPitch;
		}
		
		//Unlock the off screen surface
		hResult = pOverlayDDS->Unlock(NULL);
	}

	//Create a clipper
	hResult = pDirectDraw->CreateClipper(0, &pClipper, NULL);
	//See if the clipper was create successfully
	if (hResult != DD_OK) {
		return EC_COULD_NOT_CREATE_CLIPPER;
	}

	//Set the clipping window
	pClipper->SetHWnd(0, hWnd);
	//See if the clipping window was set successfully
	if (hResult != DD_OK) {
		//Release the clipper
		pClipper->Release();
		pClipper = NULL;

		return EC_COULD_NOT_SET_CLIPPING_WINDOW;
	}

	//Attach the clipper
	hResult = pPrimaryDDS->SetClipper(pClipper);
	//See if the clipper was attached successfully
	if (hResult != DD_OK) {
		//Release the clipper
		pClipper->Release();
		pClipper = NULL;

		return EC_COULD_NOT_ATTACH_CLIPPER;
	}

	return EC_OK;
}

VOID FreeOffScreenSurface(VOID) {
	//Detach and release the clipper
	if (pClipper != NULL) {
		//Detach the clipper
		pPrimaryDDS->SetClipper(NULL);

		//Release the clipper
		pClipper->Release();
		pClipper = NULL;
	}

	//Release the off screen surface
	if (pOverlayDDS != NULL) {
		pOverlayDDS->Release();
		pOverlayDDS = NULL;
	}

	//Release the primary surface
	if (pPrimaryDDS != NULL) {
		pPrimaryDDS->Release();
		pPrimaryDDS = NULL;
	}

	return;
}

LONG FlipOffScreenSurface(VOID) {
	HRESULT hResult;
	RECT destRect;
	DDBLTFX sDDBLTFX;
	BOOL bRet;
	POINT point;

	//Make this function safe to call if the off screen surface does not exist yet
	if (pOverlayDDS == NULL) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(DDERR_INVALIDOBJECT);

		return EC_SURFACE_DOES_NOT_EXIST;
	}

	//Set up the destination rectangle
	//Get the client rectangle
	bRet = GetClientRect(ghWnd, &destRect);
	if (bRet == FALSE) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(DDERR_GENERIC);

		return EC_COULD_NOT_GET_CLIENT_RECT;
	}

	//Convert client coordinates to screen coodrinates
	point.x = destRect.left;
	point.y = destRect.top;
	bRet = ClientToScreen(ghWnd, &point);
	if (bRet == FALSE) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(DDERR_GENERIC);

		return EC_COULD_NOT_GET_CLIENT_RECT;
	}

	//Move the client coordinates to screen coordinates
	destRect.right += point.x - destRect.left;
	destRect.bottom += point.y - destRect.top;
	destRect.left += point.x;
	destRect.top += point.y;

	//Set up the DDBLTFX structure
	sDDBLTFX.dwSize = sizeof(sDDBLTFX);
	sDDBLTFX.dwDDFX = DDBLTFX_NOTEARING;

	//Blit from the off screen surface to the primary surface
	hResult = pPrimaryDDS->Blt(&destRect, pOverlayDDS, &overlaySourceRect, DDBLT_WAIT, &sDDBLTFX);

	//See if the surface was lost
	if (hResult == DDERR_SURFACELOST) {
		//Restore the surface
		hResult = pPrimaryDDS->Restore();
		if (hResult != DD_OK) {
			//Set the DxDraw error code
			SetDxDrawErrorCode(hResult);

			return EC_BLT_FAILED;
		}

		//Try the blit again
		hResult = pPrimaryDDS->Blt(&destRect, pOverlayDDS, &overlaySourceRect, DDBLT_WAIT, &sDDBLTFX);
	}

	if (hResult != DD_OK) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(hResult);

		return EC_BLT_FAILED;
	}
	
	return EC_OK;
}

LONG LockOffScreenSurface(LPVOID *ppSurface, DWORD *pdwSurfacePitch) {
	HRESULT hResult;
	DDSURFACEDESC sDDSD;

	//Make sure the surface exists
	if (pOverlayDDS == NULL) {
		return EC_COULD_NOT_LOCK_SURFACE;
	}

	ZeroMemory(&sDDSD, sizeof(sDDSD));
	sDDSD.dwSize = sizeof(sDDSD);
	hResult = pOverlayDDS->Lock(NULL, &sDDSD, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);

	//See if the surface was lost
	if (hResult == DDERR_SURFACELOST) {
		//Restore the surface
		hResult = pOverlayDDS->Restore();
		if (hResult != DD_OK) {
			//Set the DxDraw error code
			SetDxDrawErrorCode(hResult);

			return EC_COULD_NOT_LOCK_SURFACE;
		}

		//Try to lock the surface again
		hResult = pOverlayDDS->Lock(NULL, &sDDSD, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	}

	if (hResult != DD_OK) {
		//Set the DxDraw error code
		SetDxDrawErrorCode(hResult);

		return EC_COULD_NOT_LOCK_SURFACE;
	}

	//Return data about the surface
	*ppSurface = sDDSD.lpSurface;
	*pdwSurfacePitch = sDDSD.lPitch;

	return EC_OK;
}

LONG UnlockOffScreenSurface(VOID) {
	HRESULT hResult;

	if (pOverlayDDS != NULL) {
		hResult = pOverlayDDS->Unlock(NULL);
	}

	return EC_OK;
}
